﻿namespace Eliza
{
    /// <summary>
    /// Simulates a psychiatrist.
    /// </summary>
    class Eliza : IEliza
    {
        #region Public properties

            /// <summary>
            /// Gets the message used to welcome a user to Eliza.
            /// </summary>
            public string WelcomeMessage {
                get { return "WELCOME!  PLEASE TELL ME YOUR PROBLEM."; }
            }

        #endregion

        #region Public methods

            /// <summary>
            /// Gets the response to a user's input.
            /// </summary>
            /// <param name="input">The user's input.</param>
            /// <param name="conversationHasEnded">Flag: the conversation has ended.</param>
            /// <returns>Eliza's response.</returns>
            public string GetResponse
                (string input,
                 out bool conversationHasEnded)
            {
                // Return "good bye" if the user ended the conversation
                conversationHasEnded = Eliza.DoesInputImplyEndOfConversation(input);
                if (conversationHasEnded) {
                    return Eliza.GetGoodByeResponse();
                }

                // Find matching pattern - if none, return a generic response
                object pattern = FindMatchingPatternForInput (input);
                if (pattern == null) {
                    return GetGenericResponse();
                }

                // Return an appropriate response for the pattern
                return GenerateResponseForPattern(pattern, input);
            }

        #endregion

        #region Private methods

            /// <summary>
            /// Checks if the input implies the end of the conversation.
            /// </summary>
            /// <param name="input">The input.</param>
            /// <returns>
            /// <see langword="true"/> if the input implies the end of the conversation, otherwise
            /// <see langword="false"/>.
            /// </returns>
            private static bool DoesInputImplyEndOfConversation
                (string input)
            {
                return (input.ToUpper() == "BYE") ||
                       (input.ToUpper() == "GOODBYE") ||
                       (input.ToUpper() == "QUIT");
            }

            /// <summary>
            /// Gets the goodbye response.
            /// </summary>
            /// <returns>
            /// The goodbye response.
            /// </returns>
            private static string GetGoodByeResponse()
            {
                return Eliza.GoodbyeMessage;
            }

            /// <summary>
            /// Gets a generic response.
            /// </summary>
            /// <returns>
            /// A generic response.
            /// </returns>
            private static string GetGenericResponse()
            {
                return Eliza.GenericResponse;
            }

            /// <summary>
            /// Finds the matching pattern for the user's input.
            /// </summary>
            /// <param name="input">The input.</param>
            /// <returns>The pattern or <see langword="null"/> if none was found.</returns>
            private object FindMatchingPatternForInput
                (string input)
            {
                // Stubbed for now.
                return null;
            }

            /// <summary>
            /// Generates a response for a pattern.
            /// </summary>
            /// <param name="pattern">The pattern.</param>
            /// <param name="input">The input.</param>
            /// <returns>The response.</returns>
            private string GenerateResponseForPattern
                (object pattern,
                 string input)
            {
                // Stubbed.  For now just echo what the user said.
                return string.Format("You said \"{0}\".", input);
            }

        #endregion

        #region Fields

            /// <summary>
            /// The text spoken by Eliza when the conversation has ended.
            /// </summary>
            private const string GoodbyeMessage = "GOODBYE!  IT WAS NICE TALKING WITH YOU.";

            /// <summary>
            /// A generic response by Eliza.
            /// </summary>
            private const string GenericResponse = "PLEASE GO ON.";

        #endregion
    }
}
